import re
from copy import deepcopy
from dataclasses import asdict, dataclass

import renpy.game
from char_sprites import Person
from renpy.ast import (
    Hide,
    If,
    Jump,
    Label,
    Menu,
    Python,
    Say,
    Scene,
    Show,
    UserStatement,
)
from renpy.character import ADVCharacter


def iter_speakers():
    for key, value in renpy.store.__dict__.items():
        if isinstance(value, ADVCharacter):
            yield key, value


SPEAKERS = dict(iter_speakers())


def find_speaker(character):
    for key, value in SPEAKERS.items():
        if value is character:
            return key
    return None


@dataclass
class CharacterState:
    speaker: str
    body: str
    outfit: str
    expression: str = "a_0"
    visible: bool = False


_RE_PYTHON_SWAP_BODY = re.compile(r"(\w+)\.swap_body\((\w+)\)")


def make_initial_states():
    bodies = renpy.store.bodies
    return (
        {name: CharacterState(name, name, body.default_outfit) for name, body in bodies.items()}
        | {
            (speaker := f"{name}Ghost"): CharacterState(speaker, name, body.default_outfit)
            for name, body in bodies.items()
        }
        | {
            (speaker := f"{name}puppet"): CharacterState(speaker, name, bodies[name].default_outfit)
            for name in ("john", "katrina", "kiyoshi")
        }
    )


def make_person_line_info(speaker, states):
    # When a character speaks in ghost form, the character name is used as the speaker,
    # but expressions are applied to the ghost.
    state = states.get(f"{speaker}Ghost")
    is_ghost = state and state.visible
    if not is_ghost:
        state = states[speaker]
    line_info = asdict(state)
    line_info["speaker"] = speaker
    if is_ghost:
        line_info["body"] = "ghost"
    visible = line_info.pop("visible")
    if not visible:
        # Expressions are not updated for off-screen characters.
        # Body and outfit generally are, although it is not guaranteed.
        line_info["expression"] = None
    return line_info


def make_narrator_line_info(states):
    # Generally, John is the narrator.
    line_info = make_person_line_info("john", states)
    # However, treat it as a separate speaker because it is not dialogue.
    line_info["speaker"] = "narrator"
    line_info["expression"] = None
    return line_info


LOST_SOUL = object()
"""Object used when we lost track of who is speaking."""


def make_line_info(say_node, states, prev_line_info):
    scry = say_node.scry()
    who = scry.who

    if say_node.who == "extend":
        if prev_line_info is None:
            print(f"Lost track of dialogue that is being extended on line {say_node.linenumber}")
            who = LOST_SOUL
        else:
            speaker = prev_line_info["speaker"]
            who = None if speaker == "narrator" else SPEAKERS[speaker]

    if isinstance(who, Person):
        line_info = make_person_line_info(who.image_tag, states)
    elif who is None:
        line_info = make_narrator_line_info(states)
    else:
        speaker = find_speaker(who)
        if speaker == "think":
            # John's inner voice.
            line_info = make_person_line_info("john", states)
        else:
            line_info = {"body": None, "outfit": None, "expression": None}
            if speaker is None and who is not LOST_SOUL:
                print(f"Unknown speaker {who!r} on line {say_node.linenumber}")
        line_info["speaker"] = speaker

    line_info["line"] = say_node.linenumber
    line_info["text"] = say_node.what
    return line_info


def process_node(label, states):
    # TODO: If a section has not received expressions yet, drop its expressions from the output.
    data = []
    exits = {}
    prev_line_info = None

    def emit(line_info):
        # print(" ".join(f'"{value}"' if key == "text" else f"{key}:{value}" for key, value in line_info.items()))
        data.append(line_info)

    def process_block(block, states):
        nonlocal prev_line_info
        for node in block:
            if isinstance(node, Say):
                line_info = make_line_info(node, states, prev_line_info)
                if line_info is not None:
                    emit(line_info)
                    prev_line_info = line_info
            elif isinstance(node, Show):
                what = node.imspec[0]
                state = states.get(what[0])
                if state is not None:
                    state.visible = True
                    if len(what) > 1:
                        state.expression = what[1]
            elif isinstance(node, Hide):
                what = node.imspec[0]
                state = states.get(what[0])
                if state is not None:
                    state.visible = False
            elif isinstance(node, Scene):
                for state in states.values():
                    state.visible = False
            elif isinstance(node, UserStatement):
                commands, args = node.parsed
                command = commands[0]
                if command == "resetstate":
                    states = make_initial_states()
                elif command == "outfit":
                    who, outfit = args
                    state = states[who]
                    state.outfit = outfit
                elif command == "body":
                    who, body, outfit = args
                    state = states[who]
                    state.body = body
                    if outfit is not None:
                        state.outfit = outfit
                elif command == "swap":
                    (
                        _action,
                        a_name,
                        b_name,
                        _a_ghost_emotion,
                        _b_ghost_emotion,
                        a_person_emotion,
                        b_person_emotion,
                    ) = args
                    a_state = states[a_name]
                    b_state = states[b_name]
                    a_state.body, b_state.body = b_state.body, a_state.body
                    a_state.outfit, b_state.outfit = b_state.outfit, a_state.outfit
                    a_state.expression = a_person_emotion[0]
                    b_state.expression = b_person_emotion[0]
                elif command == "clone":
                    a_name, b_name = args
                    a_state = states[a_name]
                    b_state = states[b_name]
                    a_state.body = b_state.body
                    a_state.outfit = b_state.outfit
                elif command == "morph":
                    if args[0] == "begin":
                        # The full changes take effect later, but for simplicity we already apply them.
                        (
                            _action,
                            _tf_type,
                            person_name,
                            body_name,
                            outfit,
                            emotion,
                            _relative_scale,
                            _target_ypos,
                        ) = args
                        state = states[person_name]
                        state.body = body_name
                        state.outfit = outfit
                        state.expression = emotion[0]
                elif command == "exspirit":
                    person_name, emotion = args
                    person_state = states[person_name]
                    person_state.visible = False
                    person_state.body = person_name
                    ghost_state = states[f"{person_name}Ghost"]
                    ghost_state.visible = True
                    ghost_state.expression = emotion[0]
                elif command == "possess":
                    person_name, target_name, emotion = args
                    person_state = states[person_name]
                    ghost_state = states[f"{person_name}Ghost"]
                    target_state = states[target_name]
                    person_state.body = target_state.body
                    person_state.outfit = target_state.outfit
                    person_state.visible = True
                    person_state.expression = emotion[0]
                    ghost_state.visible = False
                    if target_name != person_name:
                        target_state.visible = False
                elif command == "msg":
                    person, _og_text, text, _instant = args
                    if isinstance(text, str):
                        emit(
                            {
                                "speaker": person,
                                "body": "msg",
                                "line": node.linenumber,
                                "text": text,
                            }
                        )
                elif command == "scry":
                    if args[0] == "into":
                        # "scry into" starts a new scene.
                        for state in states.values():
                            state.visible = False
                elif command == "placeholder":
                    # Ignore text past placeholders.
                    break
                elif command == "gameover":
                    break
            elif isinstance(node, Python):
                if match := _RE_PYTHON_SWAP_BODY.match(node.code.source):
                    a_name, b_name = match.groups()
                    a_state = states[a_name]
                    b_state = states[b_name]
                    a_state.body, b_state.body = b_state.body, a_state.body
                    a_state.outfit, b_state.outfit = b_state.outfit, a_state.outfit
            elif isinstance(node, If):
                continue_states = None
                full_cover = False
                for condition, if_body in node.entries:
                    if condition == "True":
                        # This is the "else" clause; we cover all possibilities.
                        full_cover = True
                    end_states = process_block(if_body, deepcopy(states))
                    if end_states is not None:
                        # We consider the end states for all nested blocks to be equivalent,
                        # so we can pick any of them as the continuation state.
                        continue_states = end_states
                if continue_states is None:
                    # None of the nested blocks return.
                    if full_cover:
                        break
                    # Continue with the current state.
                else:
                    states = continue_states
            elif isinstance(node, Menu):
                # We consider the end states for all nested blocks to be equivalent,
                # so we can pick any of them as the continuation state.
                continue_states = None
                for label, condition, block in node.items:
                    line_info = make_narrator_line_info(states)
                    # Make a reasonable estimate of the line number.
                    line_info["line"] = block[0].linenumber - 1 if block else node.linenumber + 1
                    line_info["text"] = label
                    emit(line_info)
                    if block is not None:
                        end_states = process_block(block, deepcopy(states))
                        if end_states is not None:
                            continue_states = end_states
                if continue_states is None:
                    break
                states = continue_states
            elif isinstance(node, Jump):
                if node.expression:
                    print(f"Cannot trace computed jump on line {node.linenumber}")
                else:
                    exits[node.target] = deepcopy(states)
                break
        else:
            return states
        return None

    end_states = process_block(label.block, states)
    if end_states is not None:
        node = label.block[-1].next if label.block else label.next
        if isinstance(node, Label):
            exits[node.name] = end_states

    return data, exits


def extract_script():
    script = renpy.game.script
    todo_labels = {"day1": {}}
    done_labels = set()
    extracted = []
    while todo_labels:
        label, states = todo_labels.popitem()
        done_labels.add(label)
        node = script.lookup(label)

        print(f"Extracting: {label} in {node.filename}")
        data, exits = process_node(node, states)
        extracted.append({"label": label, "file": node.filename, "dialogue": data})

        for label, states in exits.items():
            if label in done_labels:
                continue
            todo_labels[label] = states

    return extracted
