import os
import time

from jnius import JavaException, autoclass, cast

PythonSDLActivity = autoclass("org.renpy.android.PythonSDLActivity")
mActivity = PythonSDLActivity.mActivity

Environment = autoclass("android.os.Environment")
String = autoclass("java.lang.String")
Intent = autoclass("android.content.Intent")
Activity = autoclass("android.app.Activity")
DocumentsContract = autoclass("android.provider.DocumentsContract")
ContentUris = autoclass("android.content.ContentUris")
Uri = autoclass("android.net.Uri")
Long = autoclass("java.lang.Long")
IMedia = autoclass("android.provider.MediaStore$Images$Media")
VMedia = autoclass("android.provider.MediaStore$Video$Media")
AMedia = autoclass("android.provider.MediaStore$Audio$Media")
Files = autoclass("android.provider.MediaStore$Files")
Settings = autoclass("android.provider.Settings")
Context = autoclass("android.content.Context")


if not Environment.isExternalStorageManager():
    intent = Intent(Settings.ACTION_MANAGE_APP_ALL_FILES_ACCESS_PERMISSION)
    uri = Uri.parse(f"package:{mActivity.getPackageName()}")
    intent.setData(uri)
    mActivity.startActivity(intent)


def _get_sdcard_dir():
    path = None
    context = mActivity.getApplicationContext()
    storage_manager = cast(
        "android.os.storage.StorageManager",
        context.getSystemService(Context.STORAGE_SERVICE),
    )

    if storage_manager is not None:
        storage_volumes = storage_manager.getStorageVolumes()
        for storage_volume in storage_volumes.toArray():
            if storage_volume.isRemovable():
                try:
                    directory = storage_volume.getDirectory()
                except AttributeError:
                    directory = storage_volume.getPathFile()
                path = directory.getAbsolutePath()

    return path


def _handle_external_documents(uri):
    file_id = DocumentsContract.getDocumentId(uri)
    file_type, file_name = file_id.split(":")

    primary_storage = Environment.getExternalStorageDirectory().getAbsolutePath()
    sdcard_storage = _get_sdcard_dir()

    directory = primary_storage

    if file_type == "primary":
        directory = primary_storage
    elif file_type == "home":
        directory = os.path.join(primary_storage, Environment.DIRECTORY_DOCUMENTS)
    elif sdcard_storage and file_type in sdcard_storage:
        directory = sdcard_storage

    return os.path.join(directory, file_name)


def _handle_downloads_documents(uri):
    try:
        download_dir = Environment.getExternalStoragePublicDirectory(Environment.DIRECTORY_DOWNLOADS).getPath()
        path = _parse_content(
            uri=uri,
            projection=["_display_name"],
            selection=None,
            selection_args=None,
            sort_order=None,
        )
        return os.path.join(download_dir, path)
    except Exception as e:
        print(e)

    # known locations, differ between machines
    downloads = [
        "content://downloads/public_downloads",
        "content://downloads/my_downloads",
        # all_downloads requires separate permission
        # android.permission.ACCESS_ALL_DOWNLOADS
        "content://downloads/all_downloads",
    ]

    file_id = DocumentsContract.getDocumentId(uri)
    try_uris = [ContentUris.withAppendedId(Uri.parse(down), Long.valueOf(file_id)) for down in downloads]

    # try all known Download folder uris
    # and handle JavaExceptions due to different locations
    # for content:// downloads or missing permission
    path = None
    for down in try_uris:
        try:
            path = _parse_content(
                uri=down,
                projection=["_data"],
                selection=None,
                selection_args=None,
                sort_order=None,
            )

        except JavaException:
            import traceback

            traceback.print_exc()

        # we got a path, ignore the rest
        if path:
            break

    # alternative approach to Downloads by joining
    # all data items from Activity result
    if not path:
        for down in try_uris:
            try:
                path = _parse_content(
                    uri=down,
                    projection=None,
                    selection=None,
                    selection_args=None,
                    sort_order=None,
                    index_all=True,
                )

            except JavaException:
                import traceback

                traceback.print_exc()

            # we got a path, ignore the rest
            if path:
                break
    return path


def _handle_media_documents(uri):
    file_id = DocumentsContract.getDocumentId(uri)
    file_type, file_name = file_id.split(":")
    selection = "_id=?"

    if file_type == "image":
        uri = IMedia.EXTERNAL_CONTENT_URI
    elif file_type == "video":
        uri = VMedia.EXTERNAL_CONTENT_URI
    elif file_type == "audio":
        uri = AMedia.EXTERNAL_CONTENT_URI
    # Other file type was selected (probably in the Documents folder)
    else:
        uri = Files.getContentUri("external")

    return file_name, selection, uri


def _parse_content(uri, projection, selection, selection_args, sort_order, index_all=False):
    result = None
    resolver = mActivity.getContentResolver()
    read = Intent.FLAG_GRANT_READ_URI_PERMISSION
    write = Intent.FLAG_GRANT_READ_URI_PERMISSION
    persist = Intent.FLAG_GRANT_READ_URI_PERMISSION

    # grant permission for our activity
    mActivity.grantUriPermission(mActivity.getPackageName(), uri, read | write | persist)

    if not index_all:
        cursor = resolver.query(uri, projection, selection, selection_args, sort_order)

        idx = cursor.getColumnIndex(projection[0])
        if idx != -1 and cursor.moveToFirst():
            result = cursor.getString(idx)
    else:
        result = []
        cursor = resolver.query(uri, projection, selection, selection_args, sort_order)
        while cursor.moveToNext():
            for idx in range(cursor.getColumnCount()):
                result.append(cursor.getString(idx))
        result = "/".join(result)
    return result


def resolve_activity_result_uri(uri):
    uri_authority = uri.getAuthority()
    uri_scheme = uri.getScheme().lower()

    path = None
    file_name = None
    selection = None
    downloads = None

    if uri_authority == "com.android.externalstorage.documents":
        return _handle_external_documents(uri)
    # in case a user selects a file from 'Downloads' section
    # note: this won't be triggered if a user selects a path directly
    #       e.g.: Phone -> Download -> <some file>
    elif uri_authority == "com.android.providers.downloads.documents":
        path = downloads = _handle_downloads_documents(uri)
    elif uri_authority == "com.android.providers.media.documents":
        file_name, selection, uri = _handle_media_documents(uri)

    # parse content:// scheme to path
    if uri_scheme == "content" and not downloads:
        try:
            path = _parse_content(
                uri=uri,
                projection=["_data"],
                selection=selection,
                selection_args=file_name,
                sort_order=None,
            )
        except JavaException:  # handles array error for selection_args
            path = _parse_content(
                uri=uri,
                projection=["_data"],
                selection=selection,
                selection_args=[file_name],
                sort_order=None,
            )

    # nothing to parse, file:// will return a proper path
    elif uri_scheme == "file":
        path = uri.getPath()

    return path


def show_file_picker(mime_type="application/zip", multiple=False):
    print("FP: starting")
    String = autoclass("java.lang.String")
    Intent = autoclass("android.content.Intent")
    Activity = autoclass("android.app.Activity")

    print("FP: creating intent")
    intent = Intent(Intent.ACTION_GET_CONTENT)
    intent.setType(mime_type)
    intent.addCategory(Intent.CATEGORY_OPENABLE)
    if multiple:
        intent.putExtra(Intent.EXTRA_ALLOW_MULTIPLE, True)

    print("FP: creating chooser")
    chooser = Intent.createChooser(intent, cast("java.lang.CharSequence", String("FileChooser")))

    mActivity.mActivityResultRequestCode = -1
    mActivity.mActivityResultResultCode = -1
    mActivity.mActivityResultResultData = None

    print("FP: starting activity")
    mActivity.startActivityForResult(chooser, 1)

    # poll for result
    print("FP: polling for result")
    while not (
        mActivity.mActivityResultRequestCode == 1
        or mActivity.mActivityResultResultCode != -1
        or mActivity.mActivityResultResultData is not None
    ):
        print(f"FP: poll request_code: {mActivity.mActivityResultRequestCode}")
        print(f"FP: poll result_code: {mActivity.mActivityResultResultCode}")
        print(f"FP: poll data: {mActivity.mActivityResultResultData}")
        time.sleep(0.1)

    print("FP: processing result")
    result_code = mActivity.mActivityResultResultCode
    data = mActivity.mActivityResultResultData
    print(f"FP: result_code={result_code}")
    print(f"FP: data={data}")

    print(f"FP: Enum OK: {Activity.RESULT_OK}")
    print(f"FP: Enum CANCELED: {Activity.RESULT_CANCELED}")

    if result_code == Activity.RESULT_OK:
        print("FP: result OK")
        selection = []
        try:
            for count in range(data.getClipData().getItemCount()):
                ele = resolve_activity_result_uri(data.getClipData().getItemAt(count).getUri()) or []
                selection.append(ele)
        except Exception:
            selection = [resolve_activity_result_uri(data.getData())]
        print(f"FP: result={selection}")

        return selection
    elif result_code == Activity.RESULT_CANCELED:
        print("FP: result CANCELED")
        return None
    else:
        print(f"FP: result UNKNOWN ({result_code})")
        return None
