import os
import pickle
import re
from collections import defaultdict
from itertools import product

import renpy.display.im as im
import renpy.exports as renpy
from pymage_size import get_image_size

# The list of CGs collected by add_file
cgs = {}


class CG:
    size_cache = {}

    def __init__(self, name):
        self.name = name
        self.base = None
        self.image_groups = defaultdict(dict)

    def set_base(self, filename):
        self.base = im.Image(filename)

    def add_img(self, group, attribute_name, filename):
        self.image_groups[group][attribute_name] = im.Image(filename)

    def define_images(self):
        # Get a list of all the attribute groups
        lists = [attributes.items() for attributes in self.image_groups.values()]

        for pairs in product(*lists):
            # Image name
            name = ["cg"]
            name.extend(self.name)

            # Image composition args
            args = [(0, 0), self.base]

            for attribute_name, img in sorted(pairs):
                name.append(attribute_name)

                args.append((0, 0))
                args.append(img)

            # Define image.
            name = tuple(name)
            if name in self.size_cache:
                size = self.size_cache[name]
            else:
                f = renpy.exports.file(self.base.filename)
                size = get_image_size(f).get_dimensions()
                f.close()
                self.size_cache[name] = size
            renpy.image(name, im.Composite(size, *args))


def add_file(filename):
    """Add a CG image to the list of CG entries"""
    m = re.match(r"^images/cg/(.+)/([^/]+)\.(?:.*)", filename)
    if not m:
        return

    cg_name, img_name = m.groups()
    cg_name = tuple(cg_name.split("/"))

    # Get CG object
    if cg_name in cgs:
        cg = cgs[cg_name]
    else:
        cg = CG(cg_name)
        cgs[cg_name] = cg

    # Add image to CG object
    if img_name == "base":
        cg.set_base(filename)
    else:
        group = img_name.partition("_")[0]
        cg.add_img(group, img_name, filename)


def init():
    """Finalize data and expose to Ren'Py"""
    size_cache_path = os.path.join(renpy.store.temp_dir, ".cg-size-cache.pkl")
    disk_size_cache = {}
    if renpy.store.temp_dir and os.path.isfile(size_cache_path):
        with open(size_cache_path, "rb") as f:
            disk_size_cache = pickle.load(f)
            CG.size_cache = disk_size_cache

    # Define all CG images
    for cg in cgs.values():
        cg.define_images()

    if renpy.store.temp_dir and CG.size_cache != disk_size_cache:
        with open(size_cache_path, "wb") as f:
            pickle.dump(CG.size_cache, f, protocol=pickle.HIGHEST_PROTOCOL)
